import { world, Player, Entity, EntityQueryOptions, CommandResult, Vector3, Vector2 } from "@minecraft/server";
import { clamp, getClosestEntityFromLocation, getClosestPlayerFromLocation, isMultiple, sqrDistance } from "./utils";
import { OVERWORLD, getHostPlayer } from "./main";
import { ActionFormData } from "@minecraft/server-ui";


const RECEIVED_GUIDE_BOOK_TAG = "scpd_received_cctv_guide_book";
const IS_USING_MONITOR_TAG = "scpd_is_using_cctv_monitor";
const FORM_SELECTION_INDEX_SORT_NAME_ASC = 6;
const FORM_SELECTION_INDEX_SORT_NAME_DESC = 7;
const FORM_SELECTION_INDEX_SORT_DIST_ASC = 8;
const FORM_SELECTION_INDEX_SORT_DIST_DESC = 9;
const FORM_SELECTION_INDEX_UNLINK_CAMERAS = 10;
const FORM_SELECTION_INDEX_REMOVE_MONITOR = 11;

const EQO_CCTV_MONITOR: EntityQueryOptions = {
  families: ["cctv_monitor"]
};

const EQO_CCTV_CAMERA: EntityQueryOptions = {
  families: ["cctv_camera"]
};

const EQO_CCTV_LINKER: EntityQueryOptions = {
  families: ["cctv_linker"],
  excludeTags: ["scpd_cctv_linker_chosen"]
};

const EQO_PLAYERS_USING_MONITOR: EntityQueryOptions = {
  families: ["player"],
  tags: [IS_USING_MONITOR_TAG]
};

class TempCameraLinkData {
  private _name: string;
  private _location: Vector3;
  private _entity: Entity;

  constructor(name: string, location: Vector3, entity: Entity) {
    this._name = name;
    this._location = location;
    this._entity = entity;
  }

  public get name(): string {
    return this._name;
  }

  public get location(): Vector3 {
    return this._location;
  }

  public get entity(): Entity {
    return this._entity;
  }
}


function sortCameraLinkDataArrayByDistanceAsc(cameraLinkDataArray: TempCameraLinkData[], referenceVector: Vector3): TempCameraLinkData[] {
  return cameraLinkDataArray.sort((a, b) => {
    const squaredDistanceA = sqrDistance(a.location, referenceVector);
    const squaredDistanceB = sqrDistance(b.location, referenceVector);

    // Sorting from nearest to farthest (based on squared distance)
    return squaredDistanceA - squaredDistanceB;
  });
}

function sortCameraLinkDataArrayByDistanceDesc(cameraLinkDataArray: TempCameraLinkData[], referenceVector: Vector3): TempCameraLinkData[] {
  return cameraLinkDataArray.sort((a, b) => {
    const squaredDistanceA = sqrDistance(a.location, referenceVector);
    const squaredDistanceB = sqrDistance(b.location, referenceVector);

    // Sorting from nearest to farthest (based on squared distance)
    return squaredDistanceB - squaredDistanceA;
  });
}

function exitMonitorMode(player: Player) {
  player.removeTag(IS_USING_MONITOR_TAG);
  player.runCommand("camera @s clear");
}

function monitor_getIsCameraLinkedProperty(monitorEntity: Entity, index: number): boolean {
  ++index;
  let result = monitorEntity.getProperty(`cctv_monitor:is_camera_${index}_linked`) as boolean;
  return result;
}

function monitor_setIsCameraLinkedProperty(monitorEntity: Entity, index: number, value: boolean) {
  ++index;
  monitorEntity.setProperty(`cctv_monitor:is_camera_${index}_linked`, value);
}

function monitor_getCameraPosProperty(monitorEntity: Entity, index: number): Vector3 {
  ++index;
  let x = monitorEntity.getProperty(`cctv_monitor:camera_${index}_pos_x`) as number;
  let y = monitorEntity.getProperty(`cctv_monitor:camera_${index}_pos_y`) as number;
  let z = monitorEntity.getProperty(`cctv_monitor:camera_${index}_pos_z`) as number;
  let vector3: Vector3 = { x: x, y: y, z: z }
  return vector3;
}

function monitor_setCameraPosProperty(monitorEntity: Entity, index: number, location: Vector3) {
  ++index;
  monitorEntity.setProperty(`cctv_monitor:camera_${index}_pos_x`, Math.floor(location.x));
  monitorEntity.setProperty(`cctv_monitor:camera_${index}_pos_y`, Math.floor(location.y));
  monitorEntity.setProperty(`cctv_monitor:camera_${index}_pos_z`, Math.floor(location.z));
}

function unlinkCameraFromMonitor(monitorEntity: Entity, index: number) {
  ++index;
  monitorEntity.setProperty(`cctv_monitor:is_camera_${index}_linked`, false);
  monitorEntity.setProperty(`cctv_monitor:camera_${index}_pos_x`, 0);
  monitorEntity.setProperty(`cctv_monitor:camera_${index}_pos_y`, 0);
  monitorEntity.setProperty(`cctv_monitor:camera_${index}_pos_z`, 0);
}

function unlinkAllCamerasFromMonitor(monitorEntity: Entity) {
  for (let i = 0; i < 6; i++) {
    unlinkCameraFromMonitor(monitorEntity, i);
  }
}

function runCameraCheck(posX: number, posY: number, posZ: number): CommandResult {
  posX += 0.5;
  posZ += 0.5;

  return OVERWORLD.runCommand(`testfor @e[type=lc:scpd_cctv_camera,r=0.8, x=${posX}, y=${posY}, z=${posZ}]`);
}

function runCameraCheckVector3(pos: Vector3): CommandResult {
  let x = pos.x + 0.5;
  let z = pos.z + 0.5;

  return OVERWORLD.runCommand(`testfor @e[type=lc:scpd_cctv_camera,r=0.8, x=${x}, y=${pos.y}, z=${z}]`);
}

//#region Camera list sort functions

function generateCameraLinkDdataArrayForSorting(monitorEntity: Entity): TempCameraLinkData[] {
  let cameraLinkDataArray: TempCameraLinkData[] = [];
  for (let i = 0; i < 6; i++) {
    let isCameraLinked: boolean = monitor_getIsCameraLinkedProperty(monitorEntity, i);
    if (!isCameraLinked) continue;

    let cameraLocation: Vector3 = monitor_getCameraPosProperty(monitorEntity, i);
    if (runCameraCheckVector3(cameraLocation).successCount <= 0) {
      monitor_setIsCameraLinkedProperty(monitorEntity, i, false);
      continue;
    }

    let cameraEntity: Entity | undefined = getClosestEntityFromLocation(EQO_CCTV_CAMERA, OVERWORLD, cameraLocation);
    if (cameraEntity === undefined) {
      monitor_setIsCameraLinkedProperty(monitorEntity, i, false);
      continue;
    }

    cameraLinkDataArray.push(new TempCameraLinkData(cameraEntity.nameTag, cameraLocation, cameraEntity));
  }

  return cameraLinkDataArray;
}

function completeSortingCameras(monitorEntity: Entity, cameraLinkDataArray: TempCameraLinkData[]) {
  for (let i = 0; i < 6; i++) {
    if (i >= cameraLinkDataArray.length) {
      monitor_setIsCameraLinkedProperty(monitorEntity, i, false);
      continue;
    }

    const cameraLinkData = cameraLinkDataArray[i];
    monitor_setIsCameraLinkedProperty(monitorEntity, i, true);
    monitor_setCameraPosProperty(monitorEntity, i, cameraLinkData.location);
  }
}

function sortLinkedCameras_NameAsc(monitorEntity: Entity) {
  let cameraLinkDataArray: TempCameraLinkData[] = generateCameraLinkDdataArrayForSorting(monitorEntity);
  cameraLinkDataArray.sort((a, b) => a.name.localeCompare(b.name));
  completeSortingCameras(monitorEntity, cameraLinkDataArray);
}

function sortLinkedCameras_NameDesc(monitorEntity: Entity) { 
  let cameraLinkDataArray: TempCameraLinkData[] = generateCameraLinkDdataArrayForSorting(monitorEntity);
  cameraLinkDataArray.sort((a, b) => b.name.localeCompare(a.name));
  completeSortingCameras(monitorEntity, cameraLinkDataArray);
}

function sortLinkedCameras_DistAsc(monitorEntity: Entity) { 
  let cameraLinkDataArray: TempCameraLinkData[] = generateCameraLinkDdataArrayForSorting(monitorEntity);
  sortCameraLinkDataArrayByDistanceAsc(cameraLinkDataArray, monitorEntity.location);
  completeSortingCameras(monitorEntity, cameraLinkDataArray);
}

function sortLinkedCameras_DistDesc(monitorEntity: Entity) { 
  let cameraLinkDataArray: TempCameraLinkData[] = generateCameraLinkDdataArrayForSorting(monitorEntity);
  sortCameraLinkDataArrayByDistanceDesc(cameraLinkDataArray, monitorEntity.location);
  completeSortingCameras(monitorEntity, cameraLinkDataArray);
}

//#endregion

function addCameraButtonToForm(form: ActionFormData, monitorEntity: Entity, index: number) {
  const indexPlusOne: number = index + 1;
  let isCameraLinked: boolean = monitorEntity.getProperty(`cctv_monitor:is_camera_${indexPlusOne}_linked`) as boolean;
  if (isCameraLinked) {
    let location: Vector3 = monitor_getCameraPosProperty(monitorEntity, index);
    let cameraCheckResult: CommandResult = runCameraCheck(location.x, location.y, location.z);
    if (cameraCheckResult.successCount > 0) {
      let cameraEntity: Entity = getClosestEntityFromLocation(EQO_CCTV_CAMERA, OVERWORLD, location)!;

      let buttonContent: string;
      if (cameraEntity.nameTag.length > 0) {
        buttonContent = `${cameraEntity.nameTag}`;
      } else {
        buttonContent = `X: ${location.x}, Y: ${location.y}, Z: ${location.z}`;
      }

      form.button(buttonContent);
    } else {
      monitorEntity.setProperty(`cctv_monitor:is_camera_${indexPlusOne}_linked`, false);
      form.button("§4CONNECTION LOST");
    }
  } else {
    form.button("§4CAMERA UNAVAILABLE");
  }
}

function generateMonitorInteractForm(monitorEntity: Entity): ActionFormData {
  let form = new ActionFormData();
  form.title("§lCCTV Monitor");
  form.body("Select the camera to use:");

  addCameraButtonToForm(form, monitorEntity, 0);
  addCameraButtonToForm(form, monitorEntity, 1);
  addCameraButtonToForm(form, monitorEntity, 2);
  addCameraButtonToForm(form, monitorEntity, 3);
  addCameraButtonToForm(form, monitorEntity, 4);
  addCameraButtonToForm(form, monitorEntity, 5);

  form.button("§2Sort - Name (A-Z)");
  form.button("§2Sort - Name (Z-A)");
  form.button("§2Sort - Distance (Asc)");
  form.button("§2Sort - Distance (Desc)");
  form.button("§l§cUNLINK ALL CAMERAS");
  form.button("§l§cREMOVE MONITOR");

  return form;
}

function givePlayerGuideBookIfNotYet(player: Player) {
  if (player.hasTag(RECEIVED_GUIDE_BOOK_TAG)) return;
  player.runCommand("structure load scpd:cctv_guide_book ~ ~0.1 ~");
  player.addTag(RECEIVED_GUIDE_BOOK_TAG);
  player.sendMessage("§eYou received a CCTV guide book.");
}

function onMonitorSpawned(monitorEntity: Entity) { 
  let player: Player = getClosestPlayerFromLocation(getHostPlayer(), monitorEntity.location);
  givePlayerGuideBookIfNotYet(player);
}

function onCameraSpawned(cameraEntity: Entity) { 
  let player: Player = getClosestPlayerFromLocation(getHostPlayer(), cameraEntity.location);
  givePlayerGuideBookIfNotYet(player);
}

function onMonitorEventSignalSpawned(eventSignalEntity: Entity) {
  let hostPlayer: Player | undefined = getHostPlayer();
  let playerToUseMonitor: Player = getClosestPlayerFromLocation(hostPlayer, eventSignalEntity.location);
  if (playerToUseMonitor === undefined) {
    world.sendMessage("§ccctv.js: playerToUseMonitor is undefined!")
    return;
  }

  if (!playerToUseMonitor.isValid()) {
    world.sendMessage("§ecctv.js: playerToUseMonitor is invalid. This might cause an error.");
  }

  if (playerToUseMonitor.hasTag(IS_USING_MONITOR_TAG)) {
    playerToUseMonitor.sendMessage("§cCannot use a CCTV Monitor while you are using another CCTV Monitor!");
    return;
  }

  let monitorEntity: Entity = getClosestEntityFromLocation(EQO_CCTV_MONITOR, OVERWORLD, eventSignalEntity.location) as Entity;
  if (monitorEntity === undefined) {
    world.sendMessage("§ccctv.js: monitorEntity is undefined!");
    eventSignalEntity.triggerEvent("despawn");
    return;
  }

  if (!monitorEntity.isValid()) {
    world.sendMessage("§ecctv.js: monitorEntity is invalid. This might cause an error.");
  }

  let form = generateMonitorInteractForm(monitorEntity);

  eventSignalEntity.triggerEvent("despawn");

  form.show(playerToUseMonitor).then(response => {
    if (response.canceled) {
      return;
    }

    let selectionIndex: number = response.selection!;
    switch (selectionIndex) {
      case FORM_SELECTION_INDEX_SORT_NAME_ASC:
        sortLinkedCameras_NameAsc(monitorEntity);
        return; // Sort cameras (Name, Asc)

      case FORM_SELECTION_INDEX_SORT_NAME_DESC:
        sortLinkedCameras_NameDesc(monitorEntity);
        return; // Sort cameras (Name, Desc)

      case FORM_SELECTION_INDEX_SORT_DIST_ASC:
        sortLinkedCameras_DistAsc(monitorEntity);
        return; // Sort cameras (Dist, Asc)

      case FORM_SELECTION_INDEX_SORT_DIST_DESC:
        sortLinkedCameras_DistDesc(monitorEntity);
        return; // Sort cameras (Dist, Desc)

      case FORM_SELECTION_INDEX_UNLINK_CAMERAS:
        unlinkAllCamerasFromMonitor(monitorEntity);
        return; // Unlink all cameras

      case FORM_SELECTION_INDEX_REMOVE_MONITOR:
        monitorEntity.triggerEvent("despawn");
        return; // Remove monitor

      default:
        break;
    }

    let selectionIndexPlusOne: number = selectionIndex + 1;
    let isCameraLinked: boolean = monitorEntity.getProperty(`cctv_monitor:is_camera_${selectionIndexPlusOne}_linked`) as boolean;

    if (!isCameraLinked) {
      playerToUseMonitor.sendMessage(`§cCamera ${selectionIndexPlusOne} is unavailable.`);
      return;
    }

    let location: Vector3 = monitor_getCameraPosProperty(monitorEntity, selectionIndex);
    let cameraLocation: Vector3 = { x: location.x, y: location.y, z: location.z };

    let cameraEntity: Entity | undefined = undefined;
    if (runCameraCheck(location.x, location.y, location.z).successCount > 0) {
      cameraEntity = getClosestEntityFromLocation(EQO_CCTV_CAMERA, OVERWORLD, cameraLocation) as Entity;
    }

    if (cameraEntity === undefined) {
      playerToUseMonitor.sendMessage(`§cMonitor lost connection with Camera ${selectionIndexPlusOne} (${location.x}, ${location.y}, ${location.z})`);
      monitorEntity.setProperty(`cctv_monitor:is_camera_${selectionIndexPlusOne}_linked`, false)
      return;
    }

    let cameraRotation: Vector2 = cameraEntity.getRotation();

    playerToUseMonitor.runCommand(`camera @s set minecraft:free pos ${location.x} ${location.y + 0.6} ${location.z} rot 24 ${cameraRotation.y}`);
    playerToUseMonitor.runCommand('title @s actionbar Sneak or get away from the monitor to exit');
    playerToUseMonitor.addTag(IS_USING_MONITOR_TAG);
  });
}

function onCctvLinkerSpawned(spawnedCctvLinkerEntity: Entity) {
  let monitorEntity: Entity | undefined = undefined;
  if (spawnedCctvLinkerEntity.runCommand("testfor @e[type=lc:scpd_cctv_monitor,r=1]").successCount > 0) {
    monitorEntity = getClosestEntityFromLocation(EQO_CCTV_MONITOR, OVERWORLD, spawnedCctvLinkerEntity.location);
    spawnedCctvLinkerEntity.addTag("scpd_cctv_linker_chosen");
  }

  if (monitorEntity === undefined) return;

  let cctvLinkers: Entity[] = OVERWORLD.getEntities(EQO_CCTV_LINKER);
  let newConnectedCamerasCount = 0;
  let nextCctvLinkerIndex = 0;
  let nextCameraIndex = 0;

  while (nextCameraIndex < 6) {
    if (nextCctvLinkerIndex >= cctvLinkers.length) {
      break;
    }

    if (monitorEntity.getProperty(`cctv_monitor:is_camera_${nextCameraIndex + 1}_linked`) === true) {
      nextCameraIndex++;
      world.sendMessage(`§eCamera ${nextCameraIndex} is already linked.`);
      continue;
    }

    const cctvLinker = cctvLinkers[nextCctvLinkerIndex];

    let cameraEntity: Entity | undefined;
    if (runCameraCheckVector3(cctvLinker.location).successCount > 0) {
      cameraEntity = getClosestEntityFromLocation(EQO_CCTV_CAMERA, OVERWORLD, cctvLinker.location);
    }

    if (cameraEntity === undefined) {
      world.sendMessage(`§eRemoved CCTV Linker ${nextCctvLinkerIndex + 1}: No nearby camera found.`);
      cctvLinker.triggerEvent("despawn");
      nextCctvLinkerIndex++;
      continue;
    }

    monitor_setIsCameraLinkedProperty(monitorEntity, nextCameraIndex, true);
    monitor_setCameraPosProperty(monitorEntity, nextCameraIndex, cameraEntity.location);

    cctvLinker.triggerEvent("cctv_linker:set_linked_to_true");

    newConnectedCamerasCount++;
    nextCctvLinkerIndex++;
    nextCameraIndex++;
  }

  if (newConnectedCamerasCount === 0) {
    world.sendMessage(`§eNo new CCTV cameras are linked.`);
  } else {
    world.sendMessage(`§aLinked ${newConnectedCamerasCount} CCTV cameras to the monitor!`);
  }

  spawnedCctvLinkerEntity.triggerEvent("cctv_linker:set_linked_to_true");
}

export function setupCctvSystem() {
  world.afterEvents.entitySpawn.subscribe(event => {
    switch (event.entity.typeId) {
      case "lc:scpd_cctv_monitor":
        onMonitorSpawned(event.entity);
        break;
      case "lc:scpd_cctv_camera":
        onCameraSpawned(event.entity);
        break;
      case "lc:scpd_cctv_monitor_evntsig":
        onMonitorEventSignalSpawned(event.entity);
        break;
      case "lc:scpd_cctv_linker":
        onCctvLinkerSpawned(event.entity);
        break;
      default:
        break;
    }
  });
}

export function tickCctvSystem(currentTick: number) {
  if (!isMultiple(currentTick, 10)) return;

  let playersUsingMonitor: Player[] = OVERWORLD.getPlayers(EQO_PLAYERS_USING_MONITOR);
  if (playersUsingMonitor.length === 0) return;

  playersUsingMonitor.forEach(player => {
    if (player.isSneaking) {
      exitMonitorMode(player);
      return;
    }

    if (player.runCommand("testfor @e[type=lc:scpd_cctv_monitor,r=3]").successCount <= 0) {
      exitMonitorMode(player);
      return;
    }
  });
}